/**
 * \file: mspin_logging.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * MySpin Logging
 *
 * \component: MSPIN
 *
 * \author: Thilo Bjoern Fickel ICT-ADITG/SW2 tfickel@de.adit-jv.com
 *
 * \copyright: (c) 2003 - 2013 ADIT Corporation
 *
 * \history
 * 0.1 TFickel Initial version
 *
 ***********************************************************************/

#include "mspin_logging.h"
#include <string.h>
#include <stdio.h>
#include <stdarg.h>
#ifdef MSPIN_WITH_DLT_LOGGING
#include <endian.h> //required to before including dlt.h otherwise compile error will be thrown
#include <dlt/dlt.h>
#endif //#ifdef MSPIN_WITH_DLT_LOGGING
#include <time.h>
#include <sys/time.h>
#include <errno.h>   // errno
#include <stdlib.h>  // free
#include <pthread.h> // pthread_*

#ifdef MYSPIN_WITH_TTFIS_LOGGING
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_MYSPIN_TA
#include "trcGenProj/Header/mspin_logging.c.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_MYSPIN_TA
#endif

#include "mspin_tracedefinitions.h"
#endif //MYSPIN_WITH_TTFIS_LOGGING

#define MSPIN_DBG_LVL stdout
#define MSPIN_MAX_MESSAGE_SIZE 1024 //DLT MAX is 65535, we use less
#define MSPIN_MESSAGE_TIMESTAMP_BUFFER_SIZE 64

static S32 gMspinVerbosityLevel = eMspinVerbosityDefault;
static U32 gMspinLogging = MSPIN_LOGGING_TO_DLT; //enable only DLT per default
static bool gMspinPrependTime = TRUE; //per default prepend time
static FILE* gMspinLogFile = NULL;
static char gLogFileLocation[512] = "/tmp/debugPrint.log";

//The following variable can be used to limit the max log file size. Please note that two log files will be created
// each of this size. If you want to limit the size of the log files to 1 MByte you have to specify 500kByte here.
// The second log file will be automatically created when the first one is closed because the limit is reached.
// A limit of 0 means unlimited.
static U64 gMaxFileSize = 5000000; //5 MByte

static pthread_mutex_t gFileLock = PTHREAD_MUTEX_INITIALIZER;

#ifdef MSPIN_WITH_DLT_LOGGING
static DltContext gMspinDLTContext;
static DltContext gMspinCoreDLTContext;
#endif //#ifdef MSPIN_WITH_DLT_LOGGING

static const char* mspin_log_getCurrentTime(char *pBuffer, U32 bufferLen)
{
    struct timeval tv;
    time_t nowtime;
    struct tm *nowtm;
    char tmbuf[64];

    gettimeofday(&tv, NULL);
    nowtime = tv.tv_sec;
    nowtm = localtime(&nowtime);
    strftime(tmbuf, sizeof tmbuf, "%H:%M:%S", nowtm); //omit date, only print time

    snprintf(pBuffer, bufferLen, "%s.%06d : ", tmbuf, (S32)tv.tv_usec);

    return pBuffer;
}

static bool mspin_log_openLogFile(void)
{
    //Attention: Do not use mspin_log_printLn while we have the lock. This would cause a deadlock!

    char* backupFilename = NULL;

    // Open file in append mode if not yet open
    if (!gMspinLogFile)
    {
        gMspinLogFile = fopen(gLogFileLocation, "a");

        if (!gMspinLogFile)
        {
            fprintf(stderr, "%s() ERROR: Opening file '%s' failed with '%s'(%d)\n", __FUNCTION__,
                    gLogFileLocation, strerror(errno), errno);

            return false;
        }
    }

    //Check if max file size is exceeded. If yes close the file stream and open it again in write mode
    // (which creates an empty file if already present). Before creating new, backup the original file to
    // "<filename>.bak"
    if ((gMaxFileSize > 0) && ((U32)(ftell(gMspinLogFile)) > (gMaxFileSize - 128)))
    {
        if (0 != fclose(gMspinLogFile))
        {
            fprintf(stderr, "%s() ERROR: Closing file failed with '%s'(%d)\n",
                    __FUNCTION__, strerror(errno), errno);
        }

        gMspinLogFile = NULL;

        //Create backup filename
        backupFilename = malloc(strlen(gLogFileLocation)+1+4);
        strcpy(backupFilename, gLogFileLocation);
        strcat(backupFilename, ".bak");

        //If backup is present, delete it before renaming current file
        if (access(backupFilename, F_OK) != -1)
        {
            if (0 != remove(backupFilename))
            {
                fprintf(stderr, "%s() ERROR: Deleting '%s' failed with '%s'(%d)\n",
                        __FUNCTION__, backupFilename, strerror(errno), errno);
            }
        }

        //Move current log file to backup
        if (0 != rename(gLogFileLocation, backupFilename))
        {
            fprintf(stderr, "%s() ERROR: Renaming file '%s' to '%s' failed with '%s'(%d)\n",
                    __FUNCTION__, gLogFileLocation, backupFilename, strerror(errno), errno);
        }

        //And finally open original file
        gMspinLogFile = fopen(gLogFileLocation, "w");

        free(backupFilename);
        backupFilename = NULL;

        fprintf(stderr, "%s() New log file created\n", __FUNCTION__);
    }

    return true;
}

void mspin_log_registerWithDlt(void)
{
#ifdef MSPIN_WITH_DLT_LOGGING
    DLT_REGISTER_CONTEXT(gMspinDLTContext, "MSP", "MYSPIN");
    DLT_REGISTER_CONTEXT(gMspinCoreDLTContext ,"MSPC", "MYSPIN Core");

    mspin_log_printLn(eMspinVerbosityDebug, "%s()", __FUNCTION__);
#else
    mspin_log_disableDestination(MSPIN_LOGGING_TO_DLT);
    mspin_log_printLn(eMspinVerbosityInfo, "%s() DLT is globally disabled", __FUNCTION__);
#endif //#ifdef MSPIN_WITH_DLT_LOGGING
}

void mspin_log_unregisterWithDlt(void)
{
#ifdef MSPIN_WITH_DLT_LOGGING
    DLT_UNREGISTER_CONTEXT(gMspinCoreDLTContext);
    DLT_UNREGISTER_CONTEXT(gMspinDLTContext);
    DLT_UNREGISTER_APP();

    mspin_log_printLn(eMspinVerbosityDebug, "%s()", __FUNCTION__);
#endif //#ifdef MSPIN_WITH_DLT_LOGGING
}

void mspin_log_setVerbosityLevel(S32 logLevel, bool prependTime)
{
    gMspinVerbosityLevel = logLevel;
    gMspinPrependTime = prependTime;

    mspin_log_printLn(eMspinVerbosityWarn, "%s(%d, prependTime=%s)",
            __FUNCTION__, (int)logLevel, prependTime ? "true" : "false"); //use warning level to make sure this is printed out
}

void mspin_log_setDestination(MSPIN_LOGGING_DESTINATION type, bool enable, const char* filename)
{
    switch (type)
    {
        case MSPIN_LOGGING_TO_DLT:
        case MSPIN_LOGGING_TO_TTFIS:
        case MSPIN_LOGGING_TO_STDOUT:
        case MSPIN_LOGGING_TO_FILE:
        {
            if (enable)
            {
                mspin_log_enableDestination(type);
            }
            else
            {
                mspin_log_disableDestination(type);
            }
            break;
        }
        default:
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s(type=%d, enable=%s) ERROR: Type not supported",
                    __FUNCTION__, type, enable ? "true" : "false");
            break;
        }
    }

    /* Update log filename */
    if (filename && (gMspinLogging & MSPIN_LOGGING_TO_FILE))
    {
        if (sizeof(gLogFileLocation) > strlen(filename))
        {
            int rc = 0;
            strcpy(gLogFileLocation, filename);

            //Close file if already open
            //Acquire file lock first
            rc = pthread_mutex_lock(&gFileLock);
            if (0 != rc)
            {
                fprintf(stderr, "%s() ERROR: Could not acquire lock with=%d\n", __FUNCTION__, rc);
                return;
            }

            if (gMspinLogFile)
            {
                fflush(gMspinLogFile);
                fclose(gMspinLogFile);
                /* file will be open at first log message */
                gMspinLogFile = NULL;
            }

            //Release file lock
            rc = pthread_mutex_unlock(&gFileLock); //try to unlock fileLock mutex
            if (0 != rc)
            {
                fprintf(stderr, "%s() ERROR: Could not release lock with=%d\n", __FUNCTION__, rc);
                return;
            }
        }
        else
        {
        	mspin_log_printLn(eMspinVerbosityError, "%s(type=%d, enable=%s, filename='%s') ERROR: Log filename too long",
                    __FUNCTION__, filename, enable ? "true" : "false", filename);
        }
    }
}

U32 mspin_log_getDestination(void)
{
    return gMspinLogging;
}

void mspin_log_setMaxFileSize(U64 maxFileSize)
{
    mspin_log_printLn(eMspinVerbosityInfo, "%s(maxFileSize=%lu) update from %lu",
            __FUNCTION__, maxFileSize, gMaxFileSize);
    gMaxFileSize = maxFileSize;
}

S32 mspin_log_getVerbosityLevel(void)
{
    return gMspinVerbosityLevel;
}

void mspin_log_enableDestination(MSPIN_LOGGING_DESTINATION destination)
{
    gMspinLogging |= destination;
    mspin_log_printLn(eMspinVerbosityInfo, "%s(destination=%d) logging is updated to 0x%x",
            __FUNCTION__, destination, gMspinLogging);
}

void mspin_log_disableDestination(MSPIN_LOGGING_DESTINATION destination)
{
    gMspinLogging &= ~(destination);
    mspin_log_printLn(eMspinVerbosityInfo, "%s(destination=%d) logging is updated to 0x%x",
            __FUNCTION__, destination, gMspinLogging);
}

bool mspin_log_getTimestamps(void)
{
    return gMspinPrependTime;
}

void* mspin_log_getDefaultDLTContext(void)
{
#ifdef MSPIN_WITH_DLT_LOGGING
    return &gMspinDLTContext;
#else
    return NULL;
#endif //#ifdef MSPIN_WITH_DLT_LOGGING
}

void* mspin_log_getCoreDLTContext(void)
{
#ifdef MSPIN_WITH_DLT_LOGGING
    return &gMspinCoreDLTContext;
#else
    return NULL;
#endif //#ifdef MSPIN_WITH_DLT_LOGGING
}

void mspin_log_printLn(tMspinVerbosityLevel level, const char *message, ...)
{
    int rc = 0;

    if (gMspinLogging
            && ((gMspinLogging & MSPIN_LOGGING_TO_DLT) || (gMspinVerbosityLevel >= (int)level)))
    {
        char timestamp[MSPIN_MESSAGE_TIMESTAMP_BUFFER_SIZE];
        char parsedMsg[MSPIN_MAX_MESSAGE_SIZE];
        memset(parsedMsg, 0, MSPIN_MAX_MESSAGE_SIZE);

        /* PRQA: Lint Message 530: va_list args gets initialized via va_start */
        /*lint -save -e530*/
        va_list args;
        va_start (args, message);
        vsnprintf(parsedMsg, MSPIN_MAX_MESSAGE_SIZE, message, args);
        va_end (args);
        /*lint -restore*/

#ifdef MSPIN_WITH_DLT_LOGGING
        if (gMspinLogging & MSPIN_LOGGING_TO_DLT)
        {
            /* PRQA: Lint Message 160: It's OK to suppress this warning in C code */
            /*lint -save -e160*/
            DLT_LOG(gMspinDLTContext, (DltLogLevelType)level, DLT_STRING(parsedMsg));
            /*lint -restore*/
        }
#endif //#ifdef MSPIN_WITH_DLT_LOGGING

#ifdef MYSPIN_WITH_TTFIS_LOGGING
        if (gMspinLogging & MSPIN_LOGGING_TO_TTFIS)
        {

            switch (level)
            {
               case 0:
                    ETG_TRACE_FATAL(("%s", parsedMsg));
                    break;
                case 1:
                    ETG_TRACE_ERR(("%s", parsedMsg));
                    break;
                case 2:
                    ETG_TRACE_SYS_MIN(("%s", parsedMsg));
                    break;
                case 3:
                    ETG_TRACE_SYS(("%s", parsedMsg));
                    break;
                case 4:
                    ETG_TRACE_COMP(("%s", parsedMsg));
                    break;
                case 5:
                    ETG_TRACE_USR1(("%s", parsedMsg));
                    break;
                case 6:
                    ETG_TRACE_USR2(("%s", parsedMsg));
                    break;
                case 7:
                    ETG_TRACE_USR3(("%s", parsedMsg));
                    break;
                default:
                    break;
            }
        }
#endif //MYSPIN_WITH_TTFIS_LOGGING

        if ((gMspinLogging & MSPIN_LOGGING_TO_FILE) && (gMspinVerbosityLevel >= (int)level))
        {
            //Acquire file lock
            rc = pthread_mutex_lock(&gFileLock);
            if (0 != rc)
            {
                fprintf(stderr, "%s() ERROR: Could not acquire lock with=%d\n", __FUNCTION__, rc);
                return;
            }

            if (mspin_log_openLogFile() && gMspinLogFile)
            {
                fprintf(gMspinLogFile, "%sMSPIN: %s\n",
                        gMspinPrependTime ? mspin_log_getCurrentTime(timestamp, sizeof(timestamp)) : "",
                        parsedMsg); //and append line break
                fflush(gMspinLogFile);
            }

            //Release file lock
            rc = pthread_mutex_unlock(&gFileLock);
            if (0 != rc)
            {
                fprintf(stderr, "%s() ERROR: Could not release lock with=%d\n", __FUNCTION__, rc);
                return;
            }
        }

        if ((gMspinLogging & MSPIN_LOGGING_TO_STDOUT) && (gMspinVerbosityLevel >= (int)level))
        {
            fprintf(MSPIN_DBG_LVL, "%sMSPIN: %s\n",
                    gMspinPrependTime ? mspin_log_getCurrentTime(timestamp, sizeof(timestamp)) : "",
                    parsedMsg); //and append line break
        }
    }
}

void mspin_log_printPrefixedLine(void* dltContext, tMspinVerbosityLevel level, const char *prefix, const char *message, ...)
{
#ifndef MSPIN_WITH_DLT_LOGGING
     (void)dltContext;
#endif //#ifndef MSPIN_WITH_DLT_LOGGING

     int rc = 0;

    if (gMspinLogging
            && ((gMspinLogging & MSPIN_LOGGING_TO_DLT) || (gMspinVerbosityLevel >= (int)level)))
    {
        char timestamp[MSPIN_MESSAGE_TIMESTAMP_BUFFER_SIZE];
        char parsedMsg[MSPIN_MAX_MESSAGE_SIZE];
        memset(parsedMsg, 0, MSPIN_MAX_MESSAGE_SIZE);

        /* PRQA: Lint Message 530: va_list args gets initialized via va_start */
        /*lint -save -e530*/
        va_list args;
        va_start (args, message);
        vsnprintf(parsedMsg, MSPIN_MAX_MESSAGE_SIZE, message, args);
        va_end (args);
        /*lint -restore*/

#ifdef MSPIN_WITH_DLT_LOGGING
        if (gMspinLogging & MSPIN_LOGGING_TO_DLT)
        {
            /* PRQA: Lint Message 160: It's OK to suppress this warning in C code */
            /*lint -save -e160*/
            DLT_LOG(*(DltContext*)dltContext, (DltLogLevelType)level, DLT_STRING(parsedMsg));
            /*lint -restore*/
        }
#endif //#ifdef MSPIN_WITH_DLT_LOGGING

#ifdef MYSPIN_WITH_TTFIS_LOGGING
        if (gMspinLogging & MSPIN_LOGGING_TO_TTFIS)
        {
            switch (level)
            {
               case 0:
                    ETG_TRACE_FATAL(("%s", parsedMsg));
                    break;
                case 1:
                    ETG_TRACE_ERR(("%s", parsedMsg));
                    break;
                case 2:
                    ETG_TRACE_SYS_MIN(("%s", parsedMsg));
                    break;
                case 3:
                    ETG_TRACE_SYS(("%s", parsedMsg));
                    break;
                case 4:
                    ETG_TRACE_COMP(("%s", parsedMsg));
                    break;
                case 5:
                    ETG_TRACE_USR1(("%s", parsedMsg));
                    break;
                case 6:
                    ETG_TRACE_USR2(("%s", parsedMsg));
                    break;
                case 7:
                    ETG_TRACE_USR3(("%s", parsedMsg));
                    break;
                default:
                    break;
            }
        }
#endif //MYSPIN_WITH_TTFIS_LOGGING

        if ((gMspinLogging & MSPIN_LOGGING_TO_FILE) && (gMspinVerbosityLevel >= (int)level))
        {
            //Acquire file lock
            rc = pthread_mutex_lock(&gFileLock);
            if (0 != rc)
            {
                printf("%s() ERROR: Could not acquire lock with=%d\n", __FUNCTION__, rc);
                return;
            }

            if (mspin_log_openLogFile() && gMspinLogFile)
            {
                fprintf(gMspinLogFile, "%s%s%s\n",
                        gMspinPrependTime ? mspin_log_getCurrentTime(timestamp, sizeof(timestamp)) : "",
                        prefix ? prefix : "",
                        parsedMsg); //and append line break
                fflush(gMspinLogFile);
            }

            //Release file lock
            rc = pthread_mutex_unlock(&gFileLock); //try to unlock fileLock mutex
            if (0 != rc)
            {
                printf("%s() ERROR: Could not release lock with=%d\n", __FUNCTION__, rc);
                return;
            }
        }

        if ((gMspinLogging & MSPIN_LOGGING_TO_STDOUT) && (gMspinVerbosityLevel >= (int)level))
        {
            fprintf(MSPIN_DBG_LVL, "%s%s%s\n",
                    gMspinPrependTime ? mspin_log_getCurrentTime(timestamp, sizeof(timestamp)) : "",
                    prefix ? prefix : "",
                    parsedMsg); //and append line break
        }
    }
}

void mspin_log_print(tMspinVerbosityLevel level, const char *message, ...)
{
    if (gMspinLogging
            && ((gMspinLogging & MSPIN_LOGGING_TO_DLT) || (gMspinVerbosityLevel >= (int)level)))
    {
        char parsedMsg[MSPIN_MAX_MESSAGE_SIZE];
        /* PRQA: Lint Message 530: va_list args gets initialized via va_start */
        /*lint -save -e530*/
        va_list args;
        va_start (args, message);
        vsnprintf(parsedMsg, MSPIN_MAX_MESSAGE_SIZE, message, args);
        va_end (args);
        /*lint -restore*/

#ifdef MSPIN_WITH_DLT_LOGGING
        if (gMspinLogging & MSPIN_LOGGING_TO_DLT)
        {
            /* PRQA: Lint Message 160: It's OK to suppress this warning in C code */
            /*lint -save -e160*/
            DLT_LOG(gMspinDLTContext, (DltLogLevelType)level, DLT_STRING(parsedMsg));
            /*lint -restore*/
        }
#endif //#ifdef MSPIN_WITH_DLT_LOGGING

        if ((gMspinLogging & MSPIN_LOGGING_TO_STDOUT) && (gMspinVerbosityLevel >= (int)level))
        {
            fprintf(MSPIN_DBG_LVL, "%s", parsedMsg); //and append line break
        }
    }
}

// get RSS (Resident Set Size memory usage) of own process, e.g. use like this:
// mspin_dbgPrintLine(eMspinVerbosityInfo, ">mySpin_CreateInstance RSS=%d", mspin_getRSS());
size_t mspin_log_getRSS(void)
{
    long rss = 0L;
    FILE* fp = NULL;
    if ((fp = fopen("/proc/self/statm", "r")) == NULL )
        return (size_t) 0L; // Can't open? */
    if (fscanf(fp, "%*s%ld", &rss) != 1)
    {
        fclose(fp);
        return (size_t) 0L; // Can't read? */
    }
    fclose(fp);
    return (size_t) rss * (size_t) sysconf(_SC_PAGESIZE);
}
